/*
 * Grep commands for Epsilon 5.0
 *
 * Copyright (C) 1991, K. Shane Hartman
 *
 * This file is free software; you can redistribute it and/or modify
 * it so long as this notice is preserved and no fee is charged (other than 
 * reasonable media fees) for distribution.  You are forbidden to deny these
 * rights to any other user of Epsilon.  Lugaru Software Ltd. may incorporate 
 * any and all of this code into Epsilon and have all rights to
 * the code, since it is only useful for Epsilon, although a lower level 
 * interface would work better.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
 *
 *
 * This program provides an interface to Unix like Grep program for epsilon.
 * Two commands are provided: M-x grep and M-x project grep.  grep prompts 
 * for a grep regular expression and a path spec to search in.  project-grep 
 * is similar but it will search over a list of directories (see variable 
 * project_directories below).  Output is directed to the process buffer so
 * that next-error command (C-X C-n) will take you to successive matches.
 * You can continue to edit while the search proceeds.
 *
 * Unfortunately, I have not figured out a way to notify when a match is 
 * found, so either watch the process buffer or keep trying C-x C-n.  This 
 * program assumes the existence of Gnu style Grep program in your path.
 * If you obtained this from the Lugaru bulletin board, they won't allow 
 * the grep program this goes with to reside on the BBS.  You can obtain it
 * from Channel 1 BBS in Cambridge, or by anonymous FTP from several places,
 * for example, WSMR-SIMTEL20.ARMY.MIL, FTP.AI.MIT.EDU, PREP.AI.MIT.EDU.  Just
 * look in the Gnu directories.
 *
 *
 * Revision History: for epsilon 5.0
 * Send comments, bug fixes, suggestions to shane@ai.mit.edu
 *
 * Version 1.2: 07/17/91 shane@ai.mit.edu Prepare for uploading to simtel.
 * Version 1.3: 08/02/91 shane@ai.mit.edu Changed copyright wording.  Add
 *         comments.
 * Version 1.4: 08/12/91 johnk@wrq.com (John Kercheval) Don't recenter the
 *         process buffer, remove superfluous call to make mode.  Pass
 *         the case flag to grep correctly.
 */

#include "eel.h"

/*
 * Customizable variables.
 */
#ifdef SPR

#include "vardefs.h"

#else

/* Contains the last regex searched for by grep or project-grep. */
char grep_regex[81] = "";

/*
 * A list of subdirectories or absolute directories that project-grep will 
 * search in.  It is semicolon separated list just like PATH environment 
 * variable.  If a component is not absolute, it will be interpreted as 
 * relative to project_directory variable.  For example, suppose that 
 * project_directory is "c:\project" and this variable is set to
 * ".;foo;d:\bar", then the following directories will be searched, in order,
 *
 *  the current directory
 *  c:\project\foo
 *  d:\bar
 */
char project_directories[256] = "";

/* 
 * This is the root pathname for the project grep command. Directories 
 * mentioned in project_directories are interpreted as relative to this.  It 
 * should be absolute, for example, "c:\project".
 */
char project_directory[FNAMELEN] = "";

#endif   /* SPR */

/* These are the files that grep will search in by default. */
char grep_files[81] = "*.h *.c";

/*
 * Code.
 */

do_grep (files, last, fold)
char *files;
short last;
{
  short result;
  char cmd[128];
  char *caseflag;
  char *obufname = bufname;  
  
  if (fold)
    caseflag = "-i";
  else
    caseflag = "";
  sprintf (cmd, "grep %s -n \"%s\" %s", caseflag, grep_regex, files);
  if (!another)
  {
    zap ("process");
    if (concur_shell ("", ""))
    {
      error ("Couldn't exec: error %d", errno);
    }
  }
  bufname = "process";
  point = size ();
  bprintf ("\n%s\n", cmd);
  if (last)
    bprintf ("\nexit\n");
  bufname = obufname;
}

pathname_as_directory(spec)
char *spec;
{
  int i = strlen (spec);
  if (i && !is_path_separator (spec[i - 1]))
  {
		spec[i] = path_sep, spec[i + 1] = 0;    /* add final / if none */
  }
}

/*
 * Commands.
 */

command grep()
{
  char cmd[sizeof (grep_regex)];
  
  get_strdef (cmd, "Search for", grep_regex);
  if (!*cmd)
    quick_abort ();
  strcpy (grep_regex, cmd);
  get_strdef (cmd, "in files", grep_files);
  if (!*cmd)
    quick_abort();
  strcpy (grep_files, cmd);
  do_grep (grep_files, 1, has_arg ? !case_fold : case_fold);
}

command project_grep()
{
  char cmd[sizeof (grep_regex)];
  char str[sizeof (project_directories)];
  char spec[FNAMELEN];
  char *s = str;
  char *f;
  char *ss = s;
  char *fs;
  int i;
  
  iter = 0;
  get_strdef (cmd, "Search for", grep_regex);
  if (!*cmd)
    quick_abort ();
  strcpy (grep_regex, cmd);
  get_strdef (cmd, "in files", grep_files);
  if (!*cmd)
    quick_abort ();
  strcpy (grep_files, cmd);  
  strcpy (s, project_directories);
  while (s)
  {
    ss = index (s, ';');
    if (ss)
      *ss++ = 0;
    if (strcmp (s, ".") == 0)
    {
      strcpy (spec, project_directory);
    }
    else
    {
      strcpy (spec, s);
      absolute (spec);
      if (strcmp (spec, s) != 0)
      {
        strcpy (spec, project_directory);
        pathname_as_directory (spec);
        strcat (spec, s);
      }
    }
    pathname_as_directory (spec);      
    f = grep_files;
    while (f != NULL)
    {
      strcpy (cmd, spec);
      for (i = 0; f[i] != 0 && f[i] != ' '; i++)
        ;
      if (i > 0)
      {
        char save = f[i];
        f[i] = 0;
        strcat (cmd, f);
        f[i] = save;
        for (; f[i] == ' '; i++)
          ;
        if (f[i] == 0)
          f = NULL;
        else
          f += i;
      }
      do_grep (cmd, (short) !ss && !f, has_arg ? !case_fold : case_fold);
    }
    s = ss;
  }
}
